/*****************************************************************************/
/*****************************************************************************/
/* The following files are required to run this example:                     */
/*   mixtran_macro_v2.21.sas                                                 */
/*   indivint_macro_v2.3.sas                                                 */
/*   simulated_example_data.sas7bdat                                         */
/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/
/* This example analysis uses regression calibration and fits a logistic     */
/* regression model to assess the relationship between a dietary component   */
/* and a health outcome.  For this analysis, the example data were simulated */
/* to include data from 100,000 participants, and the food frequency         */
/* questionnaire (FFQ) is the main dietary instrument.  The dietary          */
/* component is red meat, and the health outcome is a binary event variable. */
/* The simulated data include a calibration substudy of 1,000 participants   */
/* with repeated intake measurements from 24-hour dietary recalls.           */
/*                                                                           */
/* This analysis uses bootstrap variance estimation, so the univariate       */
/* measurement error model and logistic model are fit using the original     */
/* data set and 200 replicate data sets.  The following replicfirst and      */
/* repliclast macro variables allow specification of a range of replicates.  */
/* This example program considers the original data set (i.e. replicate 0).  */
/* When the replicfirst macro variable is set to 0, as it is in this         */
/* example, the 200 replicate data sets are generated using the SAS          */
/* Surveyselect procedure, and these 200 replicate data sets are combined    */
/* with the original data set and are saved for later use.                   */
/*****************************************************************************/

%let replicfirst = 0;
%let repliclast  = 0;



title1 "Fit Univariate Measurement Error Model Using MLE with FFQ as Main Instrument";
title2 "Predict Intake and Perform Regression Calibration";
title3 "Assess Relationship between a Dietary Component and a Health Outcome";



***********************************************************************;
*** The path must be modified in the following lines to specify the ***;
*** location of the SAS macros, the SAS data files, and the SAS     ***;
*** library that will be used for output                            ***;
***********************************************************************;

%let home = /prj/dcp/statprog/meas.err/develop.public.resources.stat.meth;

*** Include the required macros ***;
%include "&home/include.files.macros/mixtran_macro_v2.21.sas";
%include "&home/include.files.macros/indivint_macro_v2.3.sas";

*** Input data file ***;
libname inlib "&home/data";

*** Output data library ***;
libname outlib "&home/univar_epidemiology_example5_mle_mainffq/outlib";



***********************************************************************************;
*** Global macro variable to test for successful execution of the MIXTRAN macro ***;
***********************************************************************************;

%global success;



***********************************************************;
*** Comparison values, t0 and t1, are the 25th and 75th ***;
*** percentiles for each risk model covariate           ***;
***********************************************************;

%let t0_redmeat = 46;
%let t1_redmeat = 114;

%let t0_std_entry_age = -0.8637;
%let t1_std_entry_age =  0.8406;

%let t0_std_log_bmi = -0.6309;
%let t1_std_log_bmi =  0.6162;



*********************************************************************************;
*** The macro fit_models_replicate_loop is used to call the MIXTRAN and       ***;
*** INDIVINT macros using the original study data (i.e. replicate data set 0) ***;
*** and each of the replicate data sets for bootstrap variance estimation     ***;
*** (i.e. replicate data sets 1, 2, ..., 200)                                 ***;
*********************************************************************************;

%macro fit_models_replicate_loop;

  **************************************************************************;
  *** For replicate 0 create bootstrap data sets and save them to outlib ***;
  **************************************************************************;

  %if &replicfirst = 0 %then %do;

    *********************************************************************************;
    *** Import the simulated data file for this example and set it as replicate 0 ***;
    *********************************************************************************;

    data simdata;
      set inlib.simulated_example_data;
      replicate = 0;
    run;



    ***************************************************************************************;
    *** Create two data sets that will be the input to proc surveyselect and used to    ***;
    *** generate replicate data sets for bootstrap variance estimation.                 ***;
    *** One data set includes the calibration substudy data with 24-hour recall values  ***;
    *** and FFQ values, and the other data set includes FFQ values and no 24-hour       ***;
    *** recall values.                                                                  ***;
    ***************************************************************************************;

    data calib_substudy main_study;
      set simdata;
      if calibration_substudy = 1 then output calib_substudy;
      else output main_study;
    run;



    ********************************************************************************************************;
    *** Use proc surveyselect to generate 200 replicate bootstrap data sets for the calibration substudy ***;
    ********************************************************************************************************;

    proc surveyselect data=calib_substudy(drop = replicate) out=bootdata_calib_substudy seed=14143 method=urs sampsize=1000 outhits reps=200;
    run;



    ***************************************************************************************************;
    *** Use proc surveyselect to generate 200 replicate bootstrap data sets for the main study data ***;
    ***************************************************************************************************;

    proc surveyselect data=main_study(drop = replicate) out=bootdata_main_study seed=63491 method=urs sampsize=99000 outhits reps=200;
    run;



    ******************************************************;
    *** Combine the study data with the bootstrap data ***;
    ******************************************************;

    data simdata_and_bootdata;
      set simdata bootdata_calib_substudy bootdata_main_study;
    run;



    ************************************************************************;
    *** Create the unique record ID variable for each replicate data set ***;
    ************************************************************************;

    proc sort data=simdata_and_bootdata;
      by replicate;
    run;

    data simdata_and_bootdata;
      set simdata_and_bootdata;
      by replicate;

      retain replicaterowid;

      if first.replicate then replicaterowid = 1;
      else replicaterowid = replicaterowid + 1;
    run;



    **************************************************;
    *** Save the simulated data and bootstrap data ***;
    **************************************************;

    data outlib.simdata_and_bootdata(keep = replicate replicaterowid calibration_substudy r1_redmeat_g r2_redmeat_g
                                            std_boxcox_ffq_redmeat_g std_entry_age std_log_bmi ybin_event);
      set simdata_and_bootdata;
    run;

  %end;



  %else %do;

    *********************************************************************************;
    *** For other replicates, read in the saved simulated and bootstrap data sets ***;
    *********************************************************************************;

    data simdata_and_bootdata;
      set outlib.simdata_and_bootdata;
    run;

  %end;



  %do replicnum = &replicfirst %to &repliclast;

    title4 "Replicate &replicnum";
    data select_replicate;
      set simdata_and_bootdata (where = (replicate=&replicnum));
    run;

    **********************************************************************;
    *** Create separate data sets for the MIXTRAN and INDIVINT macros. ***;
    *** The input data has one record per person.                      ***;
    *** The MIXTRAN macro uses an input data set that includes one or  ***;
    *** more observations for each person.                             ***;
    *** The INDIVINT macro uses an input data set that includes one    ***;
    *** observation for each person.                                   ***;
    **********************************************************************;

    data calibmrec(keep = replicaterowid repeat r_redmeat_g std_boxcox_ffq_redmeat_g std_entry_age std_log_bmi)
         subj1rec(keep = replicaterowid r1_redmeat_g r2_redmeat_g std_boxcox_ffq_redmeat_g std_entry_age std_log_bmi ybin_event);

      set select_replicate;

      if calibration_substudy = 1 then do;

        ********************************************;
        *** Output record for 1st 24-hour recall ***;
        ********************************************;

        if r1_redmeat_g >= 0 then do;

          repeat = 1;

          r_redmeat_g = r1_redmeat_g;

          output calibmrec;

        end;

        ********************************************;
        *** Output record for 2nd 24-hour recall ***;
        ********************************************;

        if r2_redmeat_g >= 0 then do;

          repeat = 2;

          r_redmeat_g = r2_redmeat_g;

          output calibmrec;

        end;

      end;

      ****************************************;
      *** Output 1 record for each subject ***;
      ****************************************;

      output subj1rec;

    run;



    ************************************************************;
    *** Calculate the smallest positive 24-hour recall value ***;
    ************************************************************;

    proc univariate data=calibmrec noprint;
      where r_redmeat_g > 0;
      var r_redmeat_g;
      output out=outmin_amt min=min_amt;
    run;



    /*******************************************************************************/
    /*******************************************************************************/
    /*                                                                             */
    /*  Description of the MIXTRAN Macro                                           */
    /*                                                                             */
    /*******************************************************************************/
    /*                                                                             */
    /*                                                                             */
    /* The MIXTRAN macro is used for the analysis of episodically                  */
    /* consumed foods, foods consumed every day, and nutrients, and                */
    /* output from the MIXTRAN macro is used by the DISTRIB macro for              */
    /* estimation of the distribution of usual intake.                             */
    /*                                                                             */
    /* For episodically consumed foods, the MIXTRAN macro fits a two-              */
    /* part nonlinear mixed model where the first part considers the               */
    /* probability of consumption and the second part considers the                */
    /* consumption-day amount.  The model allows for covariates and                */
    /* includes a random effect in both parts and allows for correlation           */
    /* between the random effects (Tooze et al., 2006, Journal of the              */
    /* American Dietetic Association, 106, 1575-1587).                             */
    /*                                                                             */
    /* To fit this nonlinear mixed model with correlated random effects            */
    /* (i.e. the correlated model), starting values for the two parts of           */
    /* the model are obtained by first using the GENMOD procedure to fit           */
    /* a probability model and an amount model.  Then a nonlinear mixed            */
    /* model with uncorrelated random effects (i.e. the uncorrelated               */
    /* model) is fit using two calls to the NLMIXED procedure, and the             */
    /* parameter estimates from this model are used as starting values             */
    /* for the correlated model.                                                   */
    /*                                                                             */
    /* For foods consumed every day and nutrients, the MIXTRAN macro               */
    /* uses the GENMOD procedure to calculate starting values and uses             */
    /* the NLMIXED procedure to fit an amount-only model.                          */
    /*                                                                             */
    /* The syntax for calling the macro is:                                        */
    /*                                                                             */
    /* %mixtran(data=, response=, foodtype=, subject=, repeat=,                    */
    /*          covars_prob=, covars_amt=, outlib=, modeltype=,                    */
    /*          lambda=, replicate_var=, seq=,                                     */
    /*          weekend=, vargroup=, numvargroups=,                                */
    /*          start_val1=, start_val2=, start_val3=, vcontrol=,                  */
    /*          nloptions=, titles=, printlevel=)                                  */
    /*                                                                             */
    /*  where                                                                      */
    /*                                                                             */
    /*   "data"         * Specifies the data set to be used.                       */
    /*                                                                             */
    /*   "response"     * Specifies the 24-hour recall variable.                   */
    /*                                                                             */
    /*   "foodtype"     * Specifies a name for the analysis, used to               */
    /*                    identify the output data sets.  This value can           */
    /*                    be the same as the response variable.                    */
    /*                                                                             */
    /*   "subject"      * Specifies the variable that uniquely                     */
    /*                    identifies each subject.                                 */
    /*                                                                             */
    /*   "repeat"       * Specifies the variable that indexes repeated             */
    /*                    observations for each subject.                           */
    /*                                                                             */
    /*   "covars_prob"    Specifies a list of covariates for the first             */
    /*                    part of the model that models the probability            */
    /*                    of consumption.  Covariates must be separated            */
    /*                    by spaces.  Interactions must be in the order            */
    /*                    specified by PROC GENMOD.  If the model type             */
    /*                    is "amount" then covars_prob should be left as           */
    /*                    a null string.                                           */
    /*                                                                             */
    /*   "covars_amt"   * Specifies a list of covariates for the second            */
    /*                    part of the model that models the consumption-           */
    /*                    day amount.  Covariates must be separated by             */
    /*                    spaces.  Interactions must be in the order               */
    /*                    specified by PROC GENMOD.                                */
    /*                                                                             */
    /*   "outlib"       * Specifies a directory where output data sets             */
    /*                    are stored.  Outlib can not be null.                     */
    /*                                                                             */
    /*   "modeltype"    * Specifies the model.  The possible values are:           */
    /*                    "null string" = fit correlated model,                    */
    /*                    "corr"        = fit correlated model,                    */
    /*                    "nocorr"      = fit uncorrelated model,                  */
    /*                    "amount"      = fit amount-only model.                   */
    /*                                                                             */
    /*   "lambda"         Specifies a user supplied value for the                  */
    /*                    Box-Cox transformation parameter, lambda.  If            */
    /*                    a value is not supplied, the macro will                  */
    /*                    calculate a value for lambda.                            */
    /*                                                                             */
    /*   "replicate_var"  Specifies the variable to be used in the                 */
    /*                    replicate statement of PROC NLMIXED or the               */
    /*                    freq statement of PROC UNIVARIATE.  The                  */
    /*                    specified variable must be integer valued.               */
    /*                                                                             */
    /*   "seq"            Specifies one or more sequence indicator                 */
    /*                    variables to account for effects due to the              */
    /*                    sequence number of a subject's records.  This            */
    /*                    variable can NOT also appear in covars_prob              */
    /*                    or covars_amt.                                           */
    /*                                                                             */
    /*   "weekend"        Specifies the weekend (Fri.-Sun.) indicator              */
    /*                    variable to account for a weekend effect.  A             */
    /*                    value of 1 represents a Fri.-Sun. record, and            */
    /*                    a value of 0 represents a Mon.-Thurs. record.            */
    /*                    This variable can NOT also appear in                     */
    /*                    covars_prob or covars_amt.                               */
    /*                                                                             */
    /*   "vargroup"       Specifies a variable that groups observations            */
    /*                    to allow the model to incorporate a separate             */
    /*                    residual variance parameter for each of these            */
    /*                    groups of observations.  If the output from              */
    /*                    this macro is to be used in the DISTRIB macro,           */
    /*                    then only the weekend variable can be used.              */
    /*                                                                             */
    /*   "numvargroups"   Specifies the number of groups defined by the            */
    /*                    vargroup variable.  If the output from this              */
    /*                    macro is to be used in the DISTRIB macro and             */
    /*                    weekend is the "vargroup" variable, then the             */
    /*                    number of groups is 2.                                   */
    /*                                                                             */
    /*   "start_val1"     Starting values for probability model (nocorr).          */
    /*                    Use only when vcontrol is called and parameter           */
    /*                    estimates (i.e. _parmsf1_"foodtype") from a              */
    /*                    previous execution of an analogous model are desired.    */
    /*                    Specifies the starting values data set for the           */
    /*                    1st PROC NLMIXED (i.e. NLMIXED for probability           */
    /*                    model).                                                  */
    /*                                                                             */
    /*   "start_val2"     Starting values for the amount model.                    */
    /*                    Use only when vcontrol is called and parameter           */
    /*                    estimates (i.e. _parmsf2_"foodtype") from a              */
    /*                    previous execution of an analogous model are desired.    */
    /*                    Specifies the starting values data set for the           */
    /*                    2nd PROC NLMIXED (i.e. NLMIXED for amount                */
    /*                    model).                                                  */
    /*                                                                             */
    /*   "start_val3"     Starting values for correlated model (corr).             */
    /*                    Use only when vcontrol and parameter                     */
    /*                    estimates (i.e. _parmsf3_"foodtype") from a              */
    /*                    previous execution of an analogous model are desired.    */
    /*                    Specifies the starting values data set for the           */
    /*                    3rd PROC NLMIXED (i.e. NLMIXED for correlated            */
    /*                    model).                                                  */
    /*                                                                             */
    /*   "vcontrol"       Use only when starting values from a previous            */
    /*                    execution of the same model are also used.               */
    /*                    Specifies a 1 to 6 character name to                     */
    /*                    differentiate output data sets for runs using            */
    /*                    the same food.  See the parameters start_val1,           */
    /*                    start_val2, and start_val3.  The default is              */
    /*                    null.                                                    */
    /*                                                                             */
    /*   "nloptions"      Specifies a list of options to be added to all           */
    /*                    calls to PROC NLMIXED, for example:                      */
    /*                       nloptions=qpoints=1 gconv=1e-12 itdetails.            */
    /*                                                                             */
    /*   "titles"         Specifies the number of title lines (0-4) to             */
    /*                    be reserved for the user's titles.  Up to 4              */
    /*                    title lines may be reserved for the user's               */
    /*                    titles.  The remaining title lines are used by           */
    /*                    the macro.  The default value is 0.                      */
    /*                                                                             */
    /*   "printlevel"     Specifies 1, 2, or 3 to control the amount of            */
    /*                    information printed in the list file.                    */
    /*                    Printlevel=1 prints only the summary reports.            */
    /*                    Printlevel=2 prints summary reports and output           */
    /*                    from the NLMIXED procedures.  Printlevel=2 is            */
    /*                    the default value.  Printlevel=3 prints                  */
    /*                    summary reports and output from all of the               */
    /*                    statistical procedures.                                  */
    /*                                                                             */
    /*                                                                             */
    /* Note:  * Parameters marked with an asterisk are mandatory, so a             */
    /*          value must be supplied in the macro call.                          */
    /*                                                                             */
    /* Caution:  variable name "YN" is reserved for this macro.                    */
    /*                                                                             */
    /* Caution:  data set names "data" and "data0" and "misc_info" are             */
    /*           reserved for this macro.                                          */
    /*                                                                             */
    /*******************************************************************************/

    *****************************************************************;
    ***  Call MIXTRAN to fit a nonlinear mixed model for red meat ***;
    *****************************************************************;

    %mixtran(data          = calibmrec,
             response      = r_redmeat_g,
             foodtype      = redmeat&replicnum,
             subject       = replicaterowid,
             repeat        = repeat,
             covars_prob   = std_boxcox_ffq_redmeat_g std_entry_age std_log_bmi,
             covars_amt    = std_boxcox_ffq_redmeat_g std_entry_age std_log_bmi,
             outlib        = outlib,
             modeltype     = corr,
             lambda        = ,
             replicate_var = ,
             seq           = ,
             weekend       = ,
             vargroup      = ,
             numvargroups  = ,
             subgroup      = ,
             start_val1    = ,
             start_val2    = ,
             start_val3    = ,
             vcontrol      = ,
             nloptions     = ,
             titles        = 4,
             printlevel    = 1
             );



    ********************************************************;
    *** If the MIXTRAN macro executed successfully then  ***;
    **  continue with this iteration of the example,     ***;
    **  otherwise stop this iteration and begin the next ***;
    ********************************************************;

    %if &success = 1 %then %do;

      **********************************************************************************;
      *** Create an input data set for the INDIVINT macro to predict red meat intake ***;
      **********************************************************************************;

      data paramsubj1rec;

        if _n_ = 1 then do;
          set outlib._param_redmeat&replicnum(keep = a01_intercept a02_std_boxcox_ffq_redmeat_g a03_std_entry_age a04_std_log_bmi
                                                     p01_intercept p02_std_boxcox_ffq_redmeat_g p03_std_entry_age p04_std_log_bmi
                                                     p_var_u1 a_var_u2 cov_u1u2 a_var_e a_lambda);
          set outmin_amt(keep = min_amt);
        end;

        set subj1rec(keep = replicaterowid std_boxcox_ffq_redmeat_g std_entry_age std_log_bmi);

        *****************************************************************;
        *** Calculate the linear predictor values using the parameter ***;
        *** estimates from MIXTRAN and the covariates                 ***;
        *****************************************************************;

        x1beta1 = p01_intercept + p02_std_boxcox_ffq_redmeat_g*std_boxcox_ffq_redmeat_g +
                  p03_std_entry_age*std_entry_age + p04_std_log_bmi*std_log_bmi;

        x2beta2 = a01_intercept + a02_std_boxcox_ffq_redmeat_g*std_boxcox_ffq_redmeat_g +
                  a03_std_entry_age*std_entry_age + p04_std_log_bmi*std_log_bmi;

      run;



      /*************************************************************************/
      /*************************************************************************/
      /*                                                                       */
      /*  Description of the INDIVINT Macro                                    */
      /*                                                                       */
      /*************************************************************************/
      /*                                                                       */
      /*                                                                       */
      /* The INDIVINT macro calculates predicted values for regression         */
      /* calibration using methods from Kipnis et al. (Biometrics, 2009) and   */
      /* using results from an amount-only model or a two-part model fit using */
      /* the MIXTRAN macro.  The INDIVINT macro performs adaptive Gaussian     */
      /* quadrature to predict usual intake for each individual, and the macro */
      /* allows the user to provide a Box-Cox transformation parameter in      */
      /* order to calculate the predicted values on a transformed scale.  The  */
      /* results from this macro are intended for use in a subsequent          */
      /* regression model as discussed by Kipnis et al. (Biometrics, 2009).    */
      /*                                                                       */
      /* The syntax for calling the INDIVINT macro is:                         */
      /*                                                                       */
      /* %indivint(model12=, subj1recdata=, recid=, r24vars=, min_amt=,        */
      /*           var_u1=, var_u2=, cov_u1u2=, var_e=, lambda=, xbeta1=,      */
      /*           xbeta2=, boxcox_t_lamt=, lamt=, dencalc=, denopt=,          */
      /*           u1nlmix=, u2nlmix=, titles=, notesprt=);                    */
      /*                                                                       */
      /* where the parameters are described as follows.                        */
      /*                                                                       */
      /*  "model12"            Specifies the type of model that was fit prior  */
      /*                       to calling this macro.  A value of 1 indicates  */
      /*                       that an amount-only model was fit, and a value  */
      /*                       of 2 indicates that a two-part model was fit    */
      /*                       where part 1 is the probability part of the     */
      /*                       model and part 2 is the amount part of the      */
      /*                       model.                                          */
      /*                                                                       */
      /*  "subj1recdata"       Specifies a data set with 1 record for each     */
      /*                       individual.                                     */
      /*                                                                       */
      /*  "recid"              Specifies an identification (ID) variable that  */
      /*                       uniquely identifies each individual's record.   */
      /*                                                                       */
      /*  "r24vars"            Specifies the 24-hour recall variables with     */
      /*                       values that are either non-negative or a SAS    */
      /*                       missing value if the 24-hour recall is not      */
      /*                       available.  Variables must be space delimited   */
      /*                       as illustrated in the following example:        */
      /*                       "r24vars=r24hr1 r24hr2".                        */
      /*                       Note for Advanced Users:  If all 24-hour recall */
      /*                       values are missing for each subject, then the   */
      /*                       denominator integration should not be           */
      /*                       performed, so the "dencalc" macro parameter     */
      /*                       should be specified as "dencalc=n".             */
      /*                                                                       */
      /*  "min_amt"            Specifies a variable that provides the minimum  */
      /*                       intake amount.  This value may be selected as   */
      /*                       the smallest value among the observed           */
      /*                       consumption-day amounts.  Note that the         */
      /*                       specified variable provides the same value for  */
      /*                       each individual.  This value will be divided in */
      /*                       half and used in the calculations for the       */
      /*                       numerator integration.                          */
      /*                                                                       */
      /*  "var_u1"             Specifies a variable that provides the variance */
      /*                       estimate for u1, the random effect from the     */
      /*                       probability part of the model.  If a variable   */
      /*                       is specified, then the macro will use its value */
      /*                       as a diagonal entry of the covariance matrix    */
      /*                       which is either a 1x1 matrix or a 2x2 matrix    */
      /*                       depending on the number of random effects that  */
      /*                       are in the model.                               */
      /*                                                                       */
      /*  "var_u2"             Specifies a variable that provides the variance */
      /*                       estimate for u2, the random effect from the     */
      /*                       amount part of the model or from an amount-only */
      /*                       model.  If a variable is specified, then the    */
      /*                       macro will use its value as a diagonal entry of */
      /*                       the covariance matrix which is either a 1x1     */
      /*                       matrix or a 2x2 matrix depending on the number  */
      /*                       of random effects that are in the model.        */
      /*                                                                       */
      /*  "cov_u1u2"           Specifies a variable that provides the estimate */
      /*                       of the covariance(u1, u2) from the two-part     */
      /*                       model.  If the two-part model was an            */
      /*                       uncorrelated model, then the specified variable */
      /*                       should have a value of zero for every           */
      /*                       individual's record.                            */
      /*                                                                       */
      /*  "var_e"              Specifies a variable that provides the variance */
      /*                       estimate for e, the within-person error term    */
      /*                       from the amount part of the model or from an    */
      /*                       amount-only model.                              */
      /*                                                                       */
      /*  "lambda"             Specifies a variable that provides the estimate */
      /*                       of the Box-Cox parameter, lambda, from the      */
      /*                       amount part of the model or from an amount-only */
      /*                       model.                                          */
      /*                                                                       */
      /*  "xbeta1"             Specifies a variable that provides the linear   */
      /*                       predictor values calculated using the           */
      /*                       covariates and estimates of the fixed effects   */
      /*                       parameters from the probability part of the     */
      /*                       model.                                          */
      /*                                                                       */
      /*  "xbeta2"             Specifies a variable that provides the linear   */
      /*                       predictor values calculated using the           */
      /*                       covariates and estimates of the fixed effects   */
      /*                       parameters from the amount part of the model or */
      /*                       from an amount-only model.                      */
      /*                                                                       */
      /*  "boxcox_t_lamt"      If "boxcox_t_lamt=y" or "boxcox_t_lamt=Y" then  */
      /*                       individual usual intake will be predicted on a  */
      /*                       transformed scale where the Box-Cox             */
      /*                       transformation is used with the Box-Cox         */
      /*                       parameter value provided by the "lamt" macro    */
      /*                       parameter.  The default value for               */
      /*                       "boxcox_t_lamt" is "n".                         */
      /*                                                                       */
      /*  "lamt"               Specifies a variable that provides the Box-Cox  */
      /*                       parameter value when "boxcox_t_lamt=y" or       */
      /*                       "boxcox_t_lamt=Y".  The macro does not allow    */
      /*                       the Box-Cox parameter to be negative.           */
      /*                                                                       */
      /*  "dencalc"            By default, "dencalc=y" so the denominator      */
      /*                       integration is performed.                       */
      /*                       Note for Advanced Users:  If all 24-hour recall */
      /*                       variables are missing for each subject, then    */
      /*                       the denominator integration should not be       */
      /*                       performed, so the "dencalc" option should be    */
      /*                       specified as "dencalc=n".                       */
      /*                                                                       */
      /*  "denopt"             By default, "denopt=y" so the denominator       */
      /*                       optimization is performed as part of the        */
      /*                       denominator integration calculations.           */
      /*                       Note for Advanced Users:  In some situations    */
      /*                       the denominator optimization is redundant       */
      /*                       because the empirical Bayes estimates of u1 and */
      /*                       u2 are available from the model fitting         */
      /*                       software; therefore, in these situations,       */
      /*                       setting "denopt=n" or "denopt=N" allows the     */
      /*                       macro to skip this optimization step and use    */
      /*                       the variables provided by the "u1nlmix" and     */
      /*                       "u2nlmix" macro parameters.                     */
      /*                                                                       */
      /*  "u1nlmix"            Specifies a variable for an Advanced Users      */
      /*                       option.  For details, see the description for   */
      /*                       the "denopt" macro parameter.                   */
      /*                                                                       */
      /*  "u2nlmix"            Specifies a variable for an Advanced Users      */
      /*                       option.  For details, see the description for   */
      /*                       the "denopt" macro parameter.                   */
      /*                                                                       */
      /*  "titles"             Specifies the number of title lines to be       */
      /*                       reserved for the user's titles.  One additional */
      /*                       title line is used by the macro.  The default   */
      /*                       value is "0".                                   */
      /*                                                                       */
      /*  "notesprt"           If "notesprt=n" or "notesprt=N" then notes are  */
      /*                       not printed to the SAS log.  The default value  */
      /*                       for "notesprt" is "y".                          */
      /*                                                                       */
      /*************************************************************************/

      *******************************************************;
      *** Call INDIVINT to predict the intake of red meat ***;
      *******************************************************;

      %indivint(model12       = 2,
                subj1recdata  = paramsubj1rec,
                recid         = replicaterowid,
                r24vars       = r1_redmeat_g r2_redmeat_g,
                min_amt       = min_amt,
                var_u1        = p_var_u1,
                var_u2        = a_var_u2,
                cov_u1u2      = cov_u1u2,
                var_e         = a_var_e,
                lambda        = a_lambda,
                xbeta1        = x1beta1,
                xbeta2        = x2beta2,
                boxcox_t_lamt = ,
                lamt          = ,
                dencalc       = y,
                denopt        = y,
                u1nlmix       = ,
                u2nlmix       = ,
                titles        = 4,
                notesprt      = y
                );



      ****************************************************************;
      *** Create a data set with the INDIVINT results for red meat ***;
      ****************************************************************;

      data subj1recres;
        merge subj1rec _resdata(keep = replicaterowid indusint);
          by replicaterowid;
        rename indusint = predint_redmeat;
      run;



      *****************************************************************************************;
      *** Standardize the covariates so that one unit equals the distance between t0 and t1 ***;
      *****************************************************************************************;

      data subj1recres;
        set subj1recres;

        unit_redmeat = &t1_redmeat - &t0_redmeat;
        predint_redmeat_unit_iqr = predint_redmeat / unit_redmeat;

        unit_std_entry_age = &t1_std_entry_age - &t0_std_entry_age;
        std_entry_age_unit_iqr = std_entry_age / unit_std_entry_age;

        unit_std_log_bmi = &t1_std_log_bmi - &t0_std_log_bmi;
        std_log_bmi_unit_iqr = std_log_bmi / unit_std_log_bmi;
      run;



      ********************************************;
      *** Fit a logistic regression risk model ***;
      ********************************************;

      proc logistic data=subj1recres outest=regout;
        model ybin_event(event='1') = predint_redmeat_unit_iqr std_entry_age_unit_iqr std_log_bmi_unit_iqr;
        title5 "Logistic Regression Risk Model: Adjusted for Measurement Error";
      run;
      title5;



      *************************************************************;
      *** Save the regression results for the current replicate ***;
      *************************************************************;

      data outlib.diethealthout&replicnum(keep = replicate intercept rc_beta_redmeat_unit_iqr rc_or_redmeat_unit_iqr
                                                 rc_beta_std_entry_age_unit_iqr rc_or_std_entry_age_unit_iqr
                                                 rc_beta_std_log_bmi_unit_iqr rc_or_std_log_bmi_unit_iqr t0_redmeat t1_redmeat
                                                 t0_std_entry_age t1_std_entry_age t0_std_log_bmi t1_std_log_bmi);

        set regout(rename = (predint_redmeat_unit_iqr  = rc_beta_redmeat_unit_iqr
                             std_entry_age_unit_iqr    = rc_beta_std_entry_age_unit_iqr
                             std_log_bmi_unit_iqr      = rc_beta_std_log_bmi_unit_iqr
                             ));

        replicate = &replicnum;



        *********************************;
        *** Calculate the odds ratios ***;
        *********************************;

        rc_or_redmeat_unit_iqr       = exp(rc_beta_redmeat_unit_iqr);
        rc_or_std_entry_age_unit_iqr = exp(rc_beta_std_entry_age_unit_iqr);
        rc_or_std_log_bmi_unit_iqr   = exp(rc_beta_std_log_bmi_unit_iqr);



        ****************************************************************;
        *** Save the comparison values for each risk model covariate ***;
        ****************************************************************;

        t0_redmeat = &t0_redmeat;
        t1_redmeat = &t1_redmeat;

        t0_std_entry_age = &t0_std_entry_age;
        t1_std_entry_age = &t1_std_entry_age;

        t0_std_log_bmi = &t0_std_log_bmi;
        t1_std_log_bmi = &t1_std_log_bmi;

      run;

    ******************************************************************;
    *** End of the processing of the INDIVINT macro for iterations ***;
    *** that were successful in the MIXTRAN macro                  ***;
    ******************************************************************;

    %end;



    %else %do;

      ************************************************;
      *** Exit the loop if MIXTRAN is unsuccessful ***;
      ************************************************;

      %put MIXTRAN was not successful for replicate data set &replicnum - INDIVINT macro was not executed;

    %end;

  %end;   *** End of the replicate loop ***;

%mend fit_models_replicate_loop;



************************************************;
*** Call the fit_models_replicate_loop macro ***;
************************************************;

%fit_models_replicate_loop;
